/*
 * Pidgin - Dockapp plugin
 *
 * Copyright (C) 2008, Marek Wardzinski (emvi)
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02111-1301  USA
 *
 *
 * To start using this plugin add followed line to Wharf configuration
 * *Wharf Pidgin - Swallow "pidgin" pidgin
 *
 */

/* TODO
 * - global status don't show (when only jabber account - works)
 * */

#include "internal.h"
#include "plugin.h"
#include "connection.h"
#include "version.h"
#include "gtkplugin.h"
#include "savedstatuses.h"
#include "gtkconv.h"
#include "pidginstock.h"
#include "gtkutils.h"
#include "gtkblist.h"
#include "dockapp.h"

/* gtk/gdk globals */
GtkWidget *glDockappWindow = NULL;
GtkWidget *glContainerVBox = NULL;
GtkWidget *glContainerHBox = NULL;
GtkWidget *glIconImage[DOCKAPP_ICONS] = { NULL, NULL, NULL };
GtkWidget *glListLabel[DOCKAPP_ICONS] = { NULL, NULL, NULL };

/* Status of blinking icon */
gboolean glDockapp_blinker_icon[DOCKAPP_ICONS] = { FALSE, FALSE, FALSE };

/* Timer IDs for icon blinking */
guint glDockapp_blinker_icon_id[DOCKAPP_ICONS] = { 0, 0, 0 };

/* Counter for new message blinks */
guint glDockapp_blinker_newmsg = 0;

/* Timer ID for new message blinking */
guint glDockapp_blinker_newmsg_id = 0;

/* Buddy actual statuses */
guint glDockapp_buddy_status[DOCKAPP_BUDDIES] = {
    dockapp_offline_color,
    dockapp_offline_color,
    dockapp_offline_color
};

/* Is new authorization request received */
gboolean glAuthorizationRequest = FALSE;

/* Preferences and preferences window labels */
static const gchar *glDockapp_icons_prefs[DOCKAPP_ICONS] = {
    "/plugins/dockapp/icons/account1",
    "/plugins/dockapp/icons/account2",
    "/plugins/dockapp/icons/account3"
};
static const gchar *glDockapp_colors_prefs[DOCKAPP_COLORS] = {
    "/plugins/dockapp/colors/offline",
    "/plugins/dockapp/colors/away",
    "/plugins/dockapp/colors/online"
};
static const gchar *glDockapp_icons_labels[DOCKAPP_ICONS] = {
    N_("First icon account"),
    N_("Second icon account"),
    N_("Third icon account")
};
static const gchar *glDockapp_icons_prefs_other[DOCKAPP_ICONS_OTHER_SELECTS] = {
    N_("-not used-"),
    N_("-new message icon-"),
    N_("-standard icon-")
};
static const gchar *glDockapp_colors_labels[DOCKAPP_COLORS] = {
    N_("Buddy offline"),
    N_("Buddy online"),
    N_("Buddy away (and others)")
};

/* --- misc functions ------------------------------------------------------ */

/* return plugin handle */
void *dockapp_get_handle() {
    static gint handle;

    return &handle;
}

/* get list of new messages */
static GList *dockapp_get_new_messages() {
    GList *l_im = NULL;
    GList *l_chat = NULL;

    l_im = pidgin_conversations_find_unseen_list(PURPLE_CONV_TYPE_IM, PIDGIN_UNSEEN_TEXT, FALSE, 0);
    l_chat =
        pidgin_conversations_find_unseen_list(PURPLE_CONV_TYPE_CHAT, PIDGIN_UNSEEN_NICK, FALSE, 0);

    if ((l_im != NULL) && (l_chat != NULL))
        return g_list_concat(l_im, l_chat);
    else if (l_im != NULL)
        return l_im;
    else
        return l_chat;
}

/* update all dockapp icons */
void dockapp_update_icons_all() {
    GList *list;
    PurpleAccount *account;

    for (list = purple_accounts_get_all(); list != NULL; list = list->next) {
        account = (PurpleAccount *) list->data;
        dockapp_update_icon(account, (char *)purple_account_get_username(account),
                            purple_account_is_connecting(account));
    }
}

/* dockapp account icons
 * account- which account updated
 * account_username- name of user account
 * status- new status at account
 * connecting- is in connecting state */
static void dockapp_update_icon(PurpleAccount * account, const char *account_username,
                                gboolean connecting) {
    gchar *icon_name;
    int icon_size;
    const gchar *pref_string;
    int i;
    GList *list;
    GdkPixbuf *pb;
    PurpleStatusPrimitive status;

    /* check if are there any unseen messages */
    if ((list = dockapp_get_new_messages()) == NULL) {
        /* if new authorization */
        if (!glAuthorizationRequest) {
            /* nothing - clear timer */
            glDockapp_blinker_newmsg = 0;
            g_source_remove(glDockapp_blinker_newmsg_id);
        }
    } else
        g_list_free(list);

    /* check our current global status */
    status = purple_savedstatus_get_type(purple_savedstatus_get_current());

    /* prepare icon name and size for standard icon */
    switch (status) {
        case PURPLE_STATUS_OFFLINE:
            icon_name = PIDGIN_STOCK_TRAY_OFFLINE;
            break;
        case PURPLE_STATUS_AWAY:
            icon_name = PIDGIN_STOCK_TRAY_AWAY;
            break;
        case PURPLE_STATUS_UNAVAILABLE:
            icon_name = PIDGIN_STOCK_TRAY_BUSY;
            break;
        case PURPLE_STATUS_EXTENDED_AWAY:
            icon_name = PIDGIN_STOCK_TRAY_XA;
            break;
        case PURPLE_STATUS_INVISIBLE:
            icon_name = PIDGIN_STOCK_TRAY_INVISIBLE;
            break;
        default:
            icon_name = PIDGIN_STOCK_TRAY_AVAILABLE;
            break;
    }
    if (glDockapp_blinker_newmsg > 0)
        icon_name = PIDGIN_STOCK_TRAY_PENDING;
    if (connecting)
        icon_name = PIDGIN_STOCK_TRAY_CONNECT;
    icon_size = gtk_icon_size_from_name(PIDGIN_ICON_SIZE_TANGO_EXTRA_SMALL);

    /* show icons */
    for (i = 0; i < DOCKAPP_ICONS; ++i) {
        pref_string = purple_prefs_get_string(glDockapp_icons_prefs[i]);
        if (strcmp(pref_string, glDockapp_icons_prefs_other[2]) == 0) {
            /* 'standard icon' selected */
            gtk_image_set_from_stock(GTK_IMAGE(glIconImage[i]), icon_name, icon_size);
        } else if (strcmp(pref_string, account_username) == 0) {
            /* updated account selected */
            pb = pidgin_create_prpl_icon(account, PIDGIN_PRPL_ICON_SMALL);
            if (purple_account_is_disconnected(account) || glDockapp_blinker_icon[i])
                gdk_pixbuf_saturate_and_pixelate(pb, pb, 0.0, FALSE);
            gtk_image_set_from_pixbuf(GTK_IMAGE(glIconImage[i]), pb);
        } else if (strcmp(pref_string, glDockapp_icons_prefs_other[1]) == 0) {
            /* 'new message' selected */
            if (glDockapp_blinker_newmsg % 2)
                gtk_image_set_from_stock(GTK_IMAGE(glIconImage[i]), PIDGIN_STOCK_STATUS_MESSAGE,
                                         icon_size);
            else
                gtk_image_clear(GTK_IMAGE(glIconImage[i]));
        }
    }
}

/* draw proper buddy colors */
void dockapp_buddies_colorize() {
    GdkColor color;
    int i;

    for (i = 0; i < DOCKAPP_BUDDIES; ++i) {
        if (gdk_color_parse
            (purple_prefs_get_string(glDockapp_colors_prefs[glDockapp_buddy_status[i]]), &color))
            gtk_widget_modify_fg(glListLabel[i], GTK_STATE_NORMAL, &color);
    }
}

/* Shift buddy list up */
void dockapp_buddies_shift_up() {
    gchar *label_text;

    /* shift statuses */
    glDockapp_buddy_status[0] = glDockapp_buddy_status[1];
    glDockapp_buddy_status[1] = glDockapp_buddy_status[2];

    /* shift buddies with font settings */
    label_text =
        g_strconcat("<span font_desc=\"", purple_prefs_get_string("/plugins/dockapp/font"), "\">",
                    gtk_label_get_text(GTK_LABEL(glListLabel[1])), "</span>", NULL);
    gtk_label_set_markup(GTK_LABEL(glListLabel[0]), label_text);
    g_free(label_text);

    label_text =
        g_strconcat("<span font_desc=\"", purple_prefs_get_string("/plugins/dockapp/font"), "\">",
                    gtk_label_get_text(GTK_LABEL(glListLabel[2])), "</span>", NULL);
    gtk_label_set_markup(GTK_LABEL(glListLabel[1]), label_text);
    g_free(label_text);
}

/* --- dockapp preferences functions and data ------------------------------ */

/* callback: icon selection changed */
static void dockapp_plugin_icon_change(GtkWidget * widget, gpointer data) {
    gint icon_num = GPOINTER_TO_INT(data);  /* which icon changed */
    gint account_num = gtk_combo_box_get_active(GTK_COMBO_BOX(widget)); /* what is selected */

    if (account_num < DOCKAPP_ICONS_OTHER_SELECTS) {
        /* special selections */
        purple_prefs_set_string(glDockapp_icons_prefs[icon_num],
                                gtk_combo_box_get_active_text(GTK_COMBO_BOX(widget)));
        if (account_num <= 1)
            gtk_image_clear(GTK_IMAGE(glIconImage[icon_num]));  /* not using icon: clear */
    } else {
        /* some account selected */
        PurpleAccount *account =
            g_list_nth_data(purple_accounts_get_all(), account_num - DOCKAPP_ICONS_OTHER_SELECTS);
        if (account != NULL)
            purple_prefs_set_string(glDockapp_icons_prefs[icon_num],
                                    purple_account_get_username(account));
    }
    /* refresh icons */
    dockapp_update_icons_all();
}

/* callback: finalization of changing font dialog window */
static void dockapp_plugin_font_response(GtkDialog * font_dialog, gint response, gpointer data) {
    if (response == GTK_RESPONSE_OK) {
        gchar *fontname = NULL;
        gchar *label_text = NULL;

        fontname = gtk_font_selection_dialog_get_font_name(GTK_FONT_SELECTION_DIALOG(font_dialog));

        purple_prefs_set_string("/plugins/dockapp/font", fontname);

        /* update fonts on buddies */
        label_text =
            g_strconcat("<span font_desc=\"", fontname, "\">",
                        gtk_label_get_text(GTK_LABEL(glListLabel[0])), "</span>", NULL);
        gtk_label_set_markup(GTK_LABEL(glListLabel[0]), label_text);
        g_free(label_text);
        label_text =
            g_strconcat("<span font_desc=\"", fontname, "\">",
                        gtk_label_get_text(GTK_LABEL(glListLabel[1])), "</span>", NULL);
        gtk_label_set_markup(GTK_LABEL(glListLabel[1]), label_text);
        g_free(label_text);
        label_text =
            g_strconcat("<span font_desc=\"", fontname, "\">",
                        gtk_label_get_text(GTK_LABEL(glListLabel[2])), "</span>", NULL);
        gtk_label_set_markup(GTK_LABEL(glListLabel[2]), label_text);
        g_free(label_text);
        g_free(fontname);

    }
    gtk_widget_destroy(GTK_WIDGET(font_dialog));
}

/* callback: selected changing font */
static void dockapp_plugin_set_font(GtkWidget * widget, gpointer data) {
    GtkWidget *font_dialog = NULL;
    const gchar *pref = NULL;

    font_dialog = gtk_font_selection_dialog_new(_("Select font"));
    pref = purple_prefs_get_string("/plugins/dockapp/font");
    if ((pref != NULL) && strcmp(pref, ""))
        gtk_font_selection_set_font_name(GTK_FONT_SELECTION
                                         (GTK_FONT_SELECTION_DIALOG(font_dialog)->fontsel), pref);

    g_signal_connect(G_OBJECT(font_dialog), "response", G_CALLBACK(dockapp_plugin_font_response),
                     data);

    gtk_window_present(GTK_WINDOW(font_dialog));
}

/* callback: finalization of changing some color dialog window */
static void dockapp_plugin_color_response(GtkDialog * color_dialog, gint response, gpointer data) {
    GdkColor color;
    gchar colorstr[8];
    GtkWidget *colorsel;
    gint color_num = GPOINTER_TO_INT(data); /* which color to change */

    if (response == GTK_RESPONSE_OK) {
        /* change preferences option and update window */
        colorsel = GTK_COLOR_SELECTION_DIALOG(color_dialog)->colorsel;

        gtk_color_selection_get_current_color(GTK_COLOR_SELECTION(colorsel), &color);

        g_snprintf(colorstr, sizeof(colorstr), "#%02X%02X%02X", color.red / 256, color.green / 256,
                   color.blue / 256);

        purple_prefs_set_string(glDockapp_colors_prefs[color_num], colorstr);
        dockapp_buddies_colorize();
    }
    gtk_widget_destroy(GTK_WIDGET(color_dialog));
}

/* callback: selected changing some color */
static void dockapp_plugin_set_color(GtkWidget * widget, gpointer data) {
    GdkColor color;
    const gchar *pref = NULL;
    GtkWidget *color_dialog = NULL;
    gint color_num = GPOINTER_TO_INT(data); /* which color to change */

    /* create and show dialog window */
    color_dialog = gtk_color_selection_dialog_new(_("Select Color"));
    pref = purple_prefs_get_string(glDockapp_colors_prefs[color_num]);
    if (gdk_color_parse(pref, &color))
        gtk_color_selection_set_current_color(GTK_COLOR_SELECTION
                                              (GTK_COLOR_SELECTION_DIALOG(color_dialog)->colorsel),
                                              &color);
    g_signal_connect(G_OBJECT(color_dialog), "response", G_CALLBACK(dockapp_plugin_color_response),
                     data);

    gtk_window_present(GTK_WINDOW(color_dialog));
}

/* create preferences window */
static GtkWidget *dockapp_plugin_pref_frame(PurplePlugin * plugin) {
    GtkWidget *ret = NULL;
    GtkWidget *frame = NULL;
    GtkWidget *hbox = NULL;
    GtkWidget *label = NULL;
    GtkWidget *icon[DOCKAPP_ICONS] = { NULL, NULL, NULL };
    GtkWidget *font = NULL;
    GtkWidget *color[DOCKAPP_COLORS] = { NULL, NULL, NULL };
    GtkSizeGroup *labelsg = NULL;
    GtkSizeGroup *widgetsg = NULL;
    GList *list;
    char *account_username;
    int i, j;

    /* prepare preferences page */
    ret = gtk_vbox_new(FALSE, PIDGIN_HIG_CAT_SPACE);
    gtk_container_set_border_width(GTK_CONTAINER(ret), PIDGIN_HIG_BORDER);

    /* group widgets */
    labelsg = gtk_size_group_new(GTK_SIZE_GROUP_HORIZONTAL);
    widgetsg = gtk_size_group_new(GTK_SIZE_GROUP_HORIZONTAL);

    /* widgets to select icons */
    frame = pidgin_make_frame(ret, _("Icons"));

    for (i = 0; i < DOCKAPP_ICONS; ++i) {
        hbox = gtk_hbox_new(FALSE, PIDGIN_HIG_CAT_SPACE);
        gtk_box_pack_start(GTK_BOX(frame), hbox, FALSE, FALSE, 0);

        label = gtk_label_new(glDockapp_icons_labels[i]);
        gtk_box_pack_start(GTK_BOX(hbox), label, FALSE, FALSE, 0);
        gtk_size_group_add_widget(labelsg, label);

        icon[i] = gtk_combo_box_new_text();
        for (j = 0; j < DOCKAPP_ICONS_OTHER_SELECTS; ++j) {
            gtk_combo_box_append_text(GTK_COMBO_BOX(icon[i]), glDockapp_icons_prefs_other[j]);
            if (strcmp
                (purple_prefs_get_string(glDockapp_icons_prefs[i]),
                 glDockapp_icons_prefs_other[j]) == 0)
                gtk_combo_box_set_active(GTK_COMBO_BOX(icon[i]), j);
        }
        gtk_box_pack_start(GTK_BOX(hbox), icon[i], FALSE, FALSE, 0);
    }

    /* append all created accounts to lists */
    for (list = purple_accounts_get_all(); list != NULL; list = list->next) {
        PurpleAccount *account = (PurpleAccount *) list->data;

        account_username = (char *)purple_account_get_username(account);
        for (i = 0; i < DOCKAPP_ICONS; ++i) {
            gtk_combo_box_append_text(GTK_COMBO_BOX(icon[i]), account_username);
            if (strcmp(purple_prefs_get_string(glDockapp_icons_prefs[i]), account_username) == 0)
                gtk_combo_box_set_active(GTK_COMBO_BOX(icon[i]),
                                         g_list_position(purple_accounts_get_all(),
                                                         list) + DOCKAPP_ICONS_OTHER_SELECTS);
        }
    }
    /* connect callbacks at end */
    for (i = 0; i < DOCKAPP_ICONS; ++i) {
        g_signal_connect(G_OBJECT(icon[i]), "changed", G_CALLBACK(dockapp_plugin_icon_change),
                         GINT_TO_POINTER(i));
    }

    /* widgets to select font */
    frame = pidgin_make_frame(ret, _("Font"));
    hbox = gtk_hbox_new(FALSE, PIDGIN_HIG_CAT_SPACE);
    gtk_box_pack_start(GTK_BOX(frame), hbox, FALSE, FALSE, 0);

    label = gtk_label_new(_("Buddy list font"));
    gtk_box_pack_start(GTK_BOX(hbox), label, FALSE, FALSE, 0);
    gtk_size_group_add_widget(labelsg, label);

    font = pidgin_pixbuf_button_from_stock("", GTK_STOCK_SELECT_FONT, PIDGIN_BUTTON_HORIZONTAL);
    gtk_size_group_add_widget(widgetsg, font);
    gtk_box_pack_start(GTK_BOX(hbox), font, FALSE, FALSE, 0);
    g_signal_connect(G_OBJECT(font), "clicked", G_CALLBACK(dockapp_plugin_set_font), NULL);

    /* widgets to select colors */
    frame = pidgin_make_frame(ret, _("Colors"));

    for (i = 0; i < DOCKAPP_COLORS; ++i) {
        hbox = gtk_hbox_new(FALSE, PIDGIN_HIG_CAT_SPACE);
        gtk_box_pack_start(GTK_BOX(frame), hbox, FALSE, FALSE, 0);

        label = gtk_label_new(glDockapp_colors_labels[i]);
        gtk_box_pack_start(GTK_BOX(hbox), label, FALSE, FALSE, 0);
        gtk_size_group_add_widget(labelsg, label);

        color[i] =
            pidgin_pixbuf_button_from_stock("", GTK_STOCK_SELECT_COLOR, PIDGIN_BUTTON_HORIZONTAL);
        gtk_size_group_add_widget(widgetsg, color[i]);
        gtk_box_pack_start(GTK_BOX(hbox), color[i], FALSE, FALSE, 0);
        g_signal_connect(G_OBJECT(color[i]), "clicked", G_CALLBACK(dockapp_plugin_set_color),
                         GINT_TO_POINTER(i));
    }

    /* show all, destroy temps and return frame */
    gtk_widget_show_all(ret);
    g_object_unref(labelsg);
    g_object_unref(widgetsg);
    return ret;
}

/* info struct for dockapp preferences */
static PidginPluginUiInfo dockapp_ui_info = {
    dockapp_plugin_pref_frame,
    0,                          /* page_num (reserved) */

    /* padding */
    NULL,
    NULL,
    NULL,
    NULL
};

/* --- gtk/gdk callbacks  -------------------------------------------------- */

/* callback: mouse click on dockapp handler function */
gboolean dockapp_window_clicked(GtkWidget * widget, GdkEventButton * event, gpointer user_data) {
    GList *list;
    GList *first;

    if ((!glAuthorizationRequest) && (glDockapp_blinker_newmsg > 0)) {
        first = dockapp_get_new_messages();
        for (list = first; list != NULL; list = list->next)
            pidgin_conv_present_conversation((PurpleConversation *) list->data);
        g_list_free(first);
    } else {
        if (glAuthorizationRequest &&
            (!purple_prefs_get_bool(PIDGIN_PREFS_ROOT "/blist/list_visible")))
            glAuthorizationRequest = FALSE;
        pidgin_blist_toggle_visibility();
    }

    dockapp_update_icons_all();
    return TRUE;
}

/* callback: timer function for icon blinking */
gboolean dockapp_icon_blinker(gpointer data) {
    glDockapp_blinker_icon[GPOINTER_TO_INT(data)] =
        (!glDockapp_blinker_icon[GPOINTER_TO_INT(data)]);
    dockapp_update_icons_all();
    return TRUE;
}

/* callback: timer function for new message icon blinking */
gboolean dockapp_iconmsg_blinker(gpointer data) {
    ++glDockapp_blinker_newmsg;
    dockapp_update_icons_all();
    return glDockapp_blinker_newmsg < 7;
}

/* --- purple callbacks  --------------------------------------------------- */

/* callback: i'im trying to connect */
static void dockapp_signal_account_connecting(PurpleAccount * account, void *data) {
    int i;
    const gchar *account_username;

    account_username = purple_account_get_username(account);
    for (i = 0; i < DOCKAPP_ICONS; ++i) {
        if (strcmp(purple_prefs_get_string(glDockapp_icons_prefs[i]), account_username) == 0) {
            glDockapp_blinker_icon_id[i] =
                g_timeout_add(500, (GtkFunction) dockapp_icon_blinker, GINT_TO_POINTER(i));
            glDockapp_blinker_icon[i] = TRUE;
        }
    }
    dockapp_update_icon(account, purple_account_get_username(account), TRUE);
}

/* callback: i'am connected now */
static void dockapp_signal_signed_on(PurpleConnection * gc, void *data) {
    int i;
    const gchar *account_username;

    /* don't blink icon */
    account_username = purple_account_get_username(purple_connection_get_account(gc));
    for (i = 0; i < DOCKAPP_ICONS; ++i) {
        if (strcmp(purple_prefs_get_string(glDockapp_icons_prefs[i]), account_username) == 0) {
            g_source_remove(glDockapp_blinker_icon_id[i]);
            glDockapp_blinker_icon[i] = FALSE;
        }
    }

    dockapp_update_icon(purple_connection_get_account(gc),
                        purple_account_get_username(purple_connection_get_account(gc)), FALSE);
}

/* callback: i'am disconnected now */
static void dockapp_signal_signed_off(PurpleConnection * gc, void *data) {
    int i;
    const gchar *account_username;

    /* don't blink icon */
    account_username = purple_account_get_username(purple_connection_get_account(gc));
    for (i = 0; i < DOCKAPP_ICONS; ++i) {
        if (strcmp(purple_prefs_get_string(glDockapp_icons_prefs[i]), account_username) == 0) {
            g_source_remove(glDockapp_blinker_icon_id[i]);
            glDockapp_blinker_icon[i] = FALSE;
        }
    }

    dockapp_update_icon(purple_connection_get_account(gc),
                        purple_account_get_username(purple_connection_get_account(gc)), FALSE);
}

/* callback: my status changed */
static void dockapp_signal_savedstatus_changed(void *data) {
    dockapp_update_icons_all();
}

/* callback: someone is connected now */
static void dockapp_signal_buddy_signed_on(PurpleBuddy * buddy, void *data) {
    PurplePresence *presence;
    PurpleStatus *status;
    PurpleStatusPrimitive new_buddy_status;
    gchar *label_text;

    dockapp_buddies_shift_up();
    presence = purple_buddy_get_presence(buddy);
    status = purple_presence_get_active_status(presence);
    new_buddy_status = purple_status_type_get_primitive(purple_status_get_type(status));
    if (new_buddy_status == PURPLE_STATUS_OFFLINE) {
        glDockapp_buddy_status[2] = dockapp_offline_color;
    } else if (new_buddy_status == PURPLE_STATUS_AVAILABLE) {
        glDockapp_buddy_status[2] = dockapp_online_color;
    } else {
        glDockapp_buddy_status[2] = dockapp_away_color;
    }
    label_text =
        g_strconcat("<span font_desc=\"", purple_prefs_get_string("/plugins/dockapp/font"), "\">",
                    (buddy->alias != NULL) ? buddy->alias : buddy->name, "</span>", NULL);
    gtk_label_set_markup(GTK_LABEL(glListLabel[2]), label_text);
    g_free(label_text);
    dockapp_buddies_colorize();
}

/* callback: someone connected changed status */
static void dockapp_signal_buddy_status_changed(PurpleBuddy * buddy, PurpleStatus * old_status,
                                                PurpleStatus * status, void *data) {
    PurpleStatusPrimitive new_buddy_status;
    gchar *label_text;

    dockapp_buddies_shift_up();
    label_text =
        g_strconcat("<span font_desc=\"", purple_prefs_get_string("/plugins/dockapp/font"), "\">",
                    (buddy->alias != NULL) ? buddy->alias : buddy->name, "</span>", NULL);
    gtk_label_set_markup(GTK_LABEL(glListLabel[2]), label_text);
    g_free(label_text);
    new_buddy_status = purple_status_type_get_primitive(purple_status_get_type(status));
    if (new_buddy_status == PURPLE_STATUS_OFFLINE) {
        glDockapp_buddy_status[2] = dockapp_offline_color;
    } else if (new_buddy_status == PURPLE_STATUS_AVAILABLE) {
        glDockapp_buddy_status[2] = dockapp_online_color;
    } else {
        glDockapp_buddy_status[2] = dockapp_away_color;
    }
    dockapp_buddies_colorize();
}

/* callback: someone disconnect now */
static void dockapp_signal_buddy_signed_off(PurpleBuddy * buddy, void *data) {
    gchar *label_text;

    dockapp_buddies_shift_up();

    label_text =
        g_strconcat("<span font_desc=\"", purple_prefs_get_string("/plugins/dockapp/font"), "\">",
                    (buddy->alias != NULL) ? buddy->alias : buddy->name, "</span>", NULL);
    gtk_label_set_markup(GTK_LABEL(glListLabel[2]), label_text);
    g_free(label_text);
    glDockapp_buddy_status[2] = dockapp_offline_color;
    dockapp_buddies_colorize();
}

/* callback: someone want to authorize me (do the same like new incoming message) */
static int dockapp_signal_account_authorization_requested(PurpleAccount * account, const char *user,
                                                          gpointer data) {
    if (!purple_prefs_get_bool(PIDGIN_PREFS_ROOT "/blist/list_visible")) {
        glAuthorizationRequest = TRUE;
        glDockapp_blinker_newmsg = 1;
        dockapp_update_icons_all();
        if (glDockapp_blinker_newmsg_id)
            g_source_remove(glDockapp_blinker_newmsg_id);
        glDockapp_blinker_newmsg_id =
            g_timeout_add(500, (GtkFunction) dockapp_iconmsg_blinker, NULL);
    }
    return 0;
}

/* callback: someone or me created new conversation */
static void dockapp_signal_conversation_created(PurpleConversation * conv, void *data) {
    glDockapp_blinker_newmsg = 1;
    dockapp_update_icons_all();
    if (glDockapp_blinker_newmsg_id)
        g_source_remove(glDockapp_blinker_newmsg_id);
    glDockapp_blinker_newmsg_id = g_timeout_add(500, (GtkFunction) dockapp_iconmsg_blinker, NULL);
}

/* callback: some change in conversation */
static void dockapp_signal_conversation_updated(PurpleConversation * conv,
                                                PurpleConvUpdateType type, void *data) {
    if (type == PURPLE_CONV_UPDATE_UNSEEN) {
        glDockapp_blinker_newmsg = 1;
        dockapp_update_icons_all();
        if (glDockapp_blinker_newmsg_id)
            g_source_remove(glDockapp_blinker_newmsg_id);
        glDockapp_blinker_newmsg_id =
            g_timeout_add(500, (GtkFunction) dockapp_iconmsg_blinker, NULL);
    }
}

/* --- main plugin functions ----------------------------------------------- */

/* initialization of plugin (always run) */
static void dockapp_init(PurplePlugin * plugin) {
    /* set preferences (with default values) */
    purple_prefs_add_none("/plugins/dockapp");
    purple_prefs_add_none("/plugins/dockapp/icons");
    purple_prefs_add_string(glDockapp_icons_prefs[0], glDockapp_icons_prefs_other[0]);
    purple_prefs_add_string(glDockapp_icons_prefs[1], glDockapp_icons_prefs_other[2]);
    purple_prefs_add_string(glDockapp_icons_prefs[2], glDockapp_icons_prefs_other[0]);
    purple_prefs_add_string("/plugins/dockapp/font", "");
    purple_prefs_add_none("/plugins/dockapp/colors");
    purple_prefs_add_string(glDockapp_colors_prefs[0], DOCKAPP_DEFAULT_COLOR_OFFLINE);
    purple_prefs_add_string(glDockapp_colors_prefs[1], DOCKAPP_DEFAULT_COLOR_ONLINE);
    purple_prefs_add_string(glDockapp_colors_prefs[2], DOCKAPP_DEFAULT_COLOR_AWAY);
}

/* load plugin function */
gboolean dockapp_load(PurplePlugin * plugin) {
    int i;
    XWMHints wmhints;
    void *conn_handle = purple_connections_get_handle();
    void *blist_handle = purple_blist_get_handle();
    void *accounts_handle = purple_accounts_get_handle();
    void *conv_handle = purple_conversations_get_handle();
    void *status_handle = purple_savedstatuses_get_handle();

    /* GTK: prepare dockapp icon/window */
    /* main dockapp window */
    glDockappWindow = gtk_window_new(GTK_WINDOW_TOPLEVEL);
    gtk_widget_set_size_request(glDockappWindow, 64, 64);
    gtk_window_set_title(GTK_WINDOW(glDockappWindow), "Pidgin");
    gtk_window_set_resizable(GTK_WINDOW(glDockappWindow), FALSE);
    gtk_window_set_wmclass(GTK_WINDOW(glDockappWindow), "GM_window", "aspidgin");
    gtk_window_set_type_hint(GTK_WINDOW(glDockappWindow), GDK_WINDOW_TYPE_HINT_DOCK);
    gtk_widget_set_app_paintable(glDockappWindow, TRUE);
    gtk_widget_set_events(glDockappWindow, GDK_BUTTON_PRESS_MASK);
    g_signal_connect(G_OBJECT(glDockappWindow), "button-press-event",
                     G_CALLBACK(dockapp_window_clicked), NULL);
    gtk_widget_realize(glDockappWindow);

    /* boxes */
    glContainerVBox = gtk_vbox_new(FALSE, 0);
    gtk_container_add(GTK_CONTAINER(glDockappWindow), glContainerVBox);
    gtk_widget_realize(glContainerHBox);

    glContainerHBox = gtk_hbox_new(TRUE, 0);
    gtk_widget_set_size_request(glContainerHBox, 64, 16);
    gtk_box_pack_start(GTK_BOX(glContainerVBox), glContainerHBox, FALSE, FALSE, 0);
    gtk_widget_realize(glContainerVBox);

    /* icons */
    for (i = 0; i < DOCKAPP_ICONS; ++i) {
        glIconImage[i] = gtk_image_new();
        gtk_box_pack_start(GTK_BOX(glContainerHBox), glIconImage[i], TRUE, TRUE, 0);
        gtk_widget_realize(glIconImage[i]);
    }

    /* list of buddy status changes */
    for (i = DOCKAPP_BUDDIES - 1; i >= 0; --i) {
        glListLabel[i] = gtk_label_new("");
        gtk_label_set_justify(GTK_LABEL(glListLabel[i]), GTK_JUSTIFY_LEFT);
        gtk_box_pack_end(GTK_BOX(glContainerVBox), glListLabel[i], TRUE, TRUE, 0);
        gtk_widget_realize(glListLabel[i]);
    }

    /* set window hints */
    wmhints.initial_state = WithdrawnState;
    wmhints.flags = StateHint | IconWindowHint | IconPositionHint | WindowGroupHint;
    wmhints.icon_x = 0;
    wmhints.icon_y = 0;
    wmhints.icon_window = GDK_WINDOW_XWINDOW(glDockappWindow->window);
    wmhints.window_group = GDK_WINDOW_XWINDOW(glDockappWindow->window);

    /* show window and set hints */
    gtk_widget_show_all(glDockappWindow);
    XSetWMHints(GDK_DISPLAY(), GDK_WINDOW_XWINDOW(glDockappWindow->window), &wmhints);

    dockapp_update_icons_all();

    /* connect with signals */
    purple_signal_connect(accounts_handle, "account-connecting", plugin,
                          PURPLE_CALLBACK(dockapp_signal_account_connecting), NULL);
    purple_signal_connect(conn_handle, "signed-on", plugin,
                          PURPLE_CALLBACK(dockapp_signal_signed_on), NULL);
    purple_signal_connect(conn_handle, "signed-off", plugin,
                          PURPLE_CALLBACK(dockapp_signal_signed_off), NULL);
    purple_signal_connect(status_handle, "savedstatus-changed", plugin,
                          PURPLE_CALLBACK(dockapp_signal_savedstatus_changed), NULL);
    purple_signal_connect(blist_handle, "buddy-signed-on", plugin,
                          PURPLE_CALLBACK(dockapp_signal_buddy_signed_on), NULL);
    purple_signal_connect(blist_handle, "buddy-status-changed", plugin,
                          PURPLE_CALLBACK(dockapp_signal_buddy_status_changed), NULL);
    purple_signal_connect(blist_handle, "buddy-signed-off", plugin,
                          PURPLE_CALLBACK(dockapp_signal_buddy_signed_off), NULL);
    purple_signal_connect(accounts_handle, "account-authorization-requested", plugin,
                          PURPLE_CALLBACK(dockapp_signal_account_authorization_requested), NULL);
    purple_signal_connect(conv_handle, "conversation-created", plugin,
                          PURPLE_CALLBACK(dockapp_signal_conversation_created), NULL);
    purple_signal_connect(conv_handle, "conversation-updated", plugin,
                          PURPLE_CALLBACK(dockapp_signal_conversation_updated), NULL);

    /* can hide main window */
    pidgin_blist_visibility_manager_add();

    return TRUE;
}

/* unload plugin function
 * not manage visibility, destroy window and disconnect signals */
gboolean dockapp_unload(PurplePlugin * plugin) {
    int i;

    g_source_remove(glDockapp_blinker_newmsg_id);
    glDockapp_blinker_newmsg = 0;
    for (i = 0; i < DOCKAPP_ICONS; ++i) {
        g_source_remove(glDockapp_blinker_icon_id[i]);
        glDockapp_blinker_icon_id[i] = 0;
    }

    pidgin_blist_visibility_manager_remove();
    purple_signals_disconnect_by_handle(dockapp_get_handle());
    gtk_widget_destroy(glDockappWindow);
    return TRUE;
}

/* struct with plugin data */
static PurplePluginInfo dockapp_info = {
    PURPLE_PLUGIN_MAGIC,
    PURPLE_MAJOR_VERSION,
    PURPLE_MINOR_VERSION,
    PURPLE_PLUGIN_STANDARD,
    PIDGIN_PLUGIN_TYPE,
    0,
    NULL,                       /* GList if dependencies on other plugins */
    PURPLE_PRIORITY_DEFAULT,

    "dockapp",
    "DockApp",
    "1.0",

    N_("DockApp plugin for AfterStep"),
    N_("This plugin show Pidgin as AfterStep Wharf icon. There are three configurable icons on DockApp window and three latest buddy changes."),
    "Marek Wardzinski (emvi) <emvi at emvi dot eu dot org>",
    "http://emvi.eu.org/pidgin_dockapp/",

    dockapp_load,               /* load plugin function */
    dockapp_unload,             /* unload plugin function */
    NULL,                       /* destroy function */

    &dockapp_ui_info,           /* preferences struct */
    NULL,                       /* loader info or protocol info */
    NULL,                       /* preferences info */
    NULL,
    NULL,                       /* reserved pointer */
    NULL,                       /* reserved pointer */
    NULL,                       /* reserved pointer */
    NULL                        /* reserved pointer */
};

/* macro loading plugin */
PURPLE_INIT_PLUGIN(dockapp, dockapp_init, dockapp_info)
